package com.zteits.job.task.getfreeberths;

import com.alibaba.fastjson.JSON;
import com.clouds.common.cache.park.ParkFreeBerthsCacheUtil;
import com.xiaoleilu.hutool.util.CollectionUtil;
import com.zteits.clouds.api.apibase.constants.DataStatusEnum;
import com.zteits.clouds.api.apibase.constants.SourceTypeEnum;
import com.zteits.clouds.api.dto.order.parkorder.param.pushfreeberths.PushFreeBerthsChangeRequest;
import com.zteits.clouds.api.dto.rocketmq.datacollection.freeberths.FreeBerthsChangeMsgVO;
import com.zteits.job.dao.park.ParkFreeBerthDao;
import com.zteits.job.dao.park.ParkingLotDao;
import com.zteits.job.domain.ParkingLot;
import com.zteits.job.domain.ParkingLotExample;
import com.zteits.job.task.getfreeberths.base.ThirdFreeBerthServiceRoute;
import com.zteits.job.task.getfreeberths.param.GetRealFreeBerthsDO;
import com.zteits.job.task.getfreeberths.param.IarinParkInfo;
import com.zteits.job.task.getfreeberths.param.UpdateFreeByIrainRes;
import com.zteits.job.util.HttpClientTutorial;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.time.Duration;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 调用道闸接口，获取真实的空闲车位数
 * .<br/>
 * <p>
 * Copyright: Copyright (c) 2017  zteits
 *
 * @ClassName: GetRealFreeBerthsJob
 * @Description:
 * @version: v1.0.0
 * @author: zhaowg
 * @date: 2018年5月11日 下午5:47:42
 * Modification History:
 * Date             Author          Version            Description
 * ---------------------------------------------------------*
 * 2018年5月11日      zhaowg           v1.0.0               创建
 */

@Component
public class PublicDealFreeBerths {
    private static final Logger logger = LoggerFactory.getLogger(PublicDealFreeBerths.class);
    @Autowired
    private ParkingLotDao parkingLotDao;
    @Autowired
    private ParkFreeBerthDao parkFreeBerthDao;
    @Autowired
    private Map<String, CallThirdQueryFreeBerthService> thirdBerthServiceMap;


    @Async("myTaskAsyncPool")
    public void dealFreeBerths(ParkingLot parkingLotDTO) {
        if (99 == parkingLotDTO.getSourceType().intValue()) {
            return;
        }
        logger.info("开始查询停车场" + parkingLotDTO.getPlName() + "[" + parkingLotDTO.getPlNo() + "]");
        String plNo = parkingLotDTO.getPlNo();
        GetRealFreeBerthsDO getRealFreeBerthsDO = queryFreeBerths(parkingLotDTO);
        if (getRealFreeBerthsDO == null) {
            return;
        }
        //调用接口返回的真实空闲车位数，可能为负数或者超过总车位数
        Integer realFreeBerths = StringUtils.isBlank(getRealFreeBerthsDO.getFreeParkingSpace()) ? 0 : Integer.valueOf(getRealFreeBerthsDO.getFreeParkingSpace());
        //修改后的空闲车位数，最小0，最大不超过总车位数
        Integer freeBerths = realFreeBerths;
        //判断数据有效性
        if (freeBerths < 0) {
            logger.info("返回的空闲车位数小于0，修改为0");
            freeBerths = 0;
        } else if (freeBerths > parkingLotDTO.getPlBerthNum()) {
            logger.info("返回的空闲车位数大于总车位数" + parkingLotDTO.getPlBerthNum() + "，修改为总车位数");
            freeBerths = parkingLotDTO.getPlBerthNum();
        }
        //判断缓存中空闲车位数是否发生变化了
        FreeBerthsChangeMsgVO freeBerthsChangeMsgVO = ParkFreeBerthsCacheUtil.getFreeBerthsObjectByPlNo(plNo);
        if (freeBerthsChangeMsgVO == null) {
            return;
        }
        if (freeBerthsChangeMsgVO.getRealFreeBerths() != realFreeBerths) {
            logger.info(plNo + "-原来真实空闲车位数：" + freeBerthsChangeMsgVO.getRealFreeBerths() + "，新真实空闲车位数：" + realFreeBerths + ",新空闲车位数：" + freeBerths);
            //更新缓存
            freeBerthsChangeMsgVO = ParkFreeBerthsCacheUtil.setFreeBerthsByPlNo(plNo, freeBerths, realFreeBerths);
        }
        if (SourceTypeEnum.NAN_ZHE.getValue().equals(parkingLotDTO.getSourceType())) {
            //南泽的总车位数不正确
            return;
        }
        //判断总车位数是否一致
        Integer plBerthNum = StringUtils.isBlank(getRealFreeBerthsDO.getTotalParkingSpace()) ? 0 : Integer.valueOf(getRealFreeBerthsDO.getTotalParkingSpace());
        if (!plBerthNum.equals(parkingLotDTO.getPlBerthNum())
                && plBerthNum.intValue() > 0) {
            logger.debug(parkingLotDTO.getPlName() + ",总车位数:" + parkingLotDTO.getPlBerthNum() + "与接口返回的不一致，更新总车位数为" + plBerthNum);
            parkingLotDao.updateTotalBerths(plNo, plBerthNum);
        }

        List<FreeBerthsChangeMsgVO> msgVOList = new ArrayList<>();
        msgVOList.add(freeBerthsChangeMsgVO);
        PushFreeBerthsChangeRequest pushFreeBerthsChangeRequest = new PushFreeBerthsChangeRequest();
        pushFreeBerthsChangeRequest.setBerthsChangeMsgVOs(msgVOList);
        pushFreeBerthsChangeRequest.setSysCode("XXL-JOB");
        //更新空闲车位数
        parkFreeBerthDao.updateFreeBerthsByPlNo(pushFreeBerthsChangeRequest);
    }

    /**
     * 调用第三方接口查询空闲车位数
     *
     * @throws IOException 2018年5月11日 zhaowg
     */
    private GetRealFreeBerthsDO queryFreeBerths(ParkingLot parkingLot) {
        try {
            /**根据数据来源路由具体的通知服务*/
            CallThirdQueryFreeBerthService queryThirdBerthInfoService = ThirdFreeBerthServiceRoute
                    .selectThirdBerthService(SourceTypeEnum.getEnumByValue(parkingLot.getSourceType()), thirdBerthServiceMap);
            /**根据数据来源类型，查询第三方空闲车位数*/
            return queryThirdBerthInfoService.queryFreeBerths(parkingLot);
        } catch (Exception e) {
            logger.error("处理错误", e);
        }
        return null;
    }

}
