package com.zteits.job.dao.park.impl;

import com.alibaba.druid.util.StringUtils;
import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.clouds.common.cache.park.ParkFreeBerthsCacheUtil;
import com.clouds.common.cache.sys.SysCodeValueCacheUtil;
import com.clouds.common.constants.CodeValKindEnum;
import com.zteits.clouds.api.apibase.constants.DataStatusEnum;
import com.zteits.clouds.api.apibase.constants.ErrorType;
import com.zteits.clouds.api.apibase.exception.BizException;
import com.zteits.clouds.api.dto.order.parkorder.param.pushfreeberths.InductionRelease;
import com.zteits.clouds.api.dto.order.parkorder.param.pushfreeberths.InductionResponse;
import com.zteits.clouds.api.dto.order.parkorder.param.pushfreeberths.PushFreeBerthsChangeRequest;
import com.zteits.clouds.api.dto.order.parkorder.param.pushfreeberths.PushFreeBerthsRule;
import com.zteits.clouds.api.dto.rocketmq.constants.MQTopicConstants;
import com.zteits.clouds.api.dto.rocketmq.constants.tags.DataCollectionTags;
import com.zteits.clouds.api.dto.rocketmq.datacollection.freeberths.FreeBerthsChangeMsgVO;
import com.zteits.job.dao.park.BerthsDao;
import com.zteits.job.dao.park.FreeBerthStatisticRuleDao;
import com.zteits.job.dao.park.ParkFreeBerthDao;
import com.zteits.job.dao.park.ParkingLotDao;
import com.zteits.job.domain.ParkFreeBerth;
import com.zteits.job.domain.ParkFreeBerthExample;
import com.zteits.job.domain.ParkingLot;
import com.zteits.job.domain.TpPFreeberthStatisticRule;
import com.zteits.job.mapper.ParkFreeBerthMapper;
import com.zteits.job.util.HttpClientTutorial;
import com.zteits.job.util.HttpUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Component
public class ParkFreeBerthDaoImpl implements ParkFreeBerthDao {
    private static final Logger logger = LoggerFactory.getLogger(ParkFreeBerthDaoImpl.class);
    @Autowired
    private FreeBerthStatisticRuleDao statisticRuleDao;
    @Autowired
    private ParkFreeBerthMapper parkFreeBerthMapper;
    @Autowired
    private ParkingLotDao parkingLotDao;
    @Autowired
    private BerthsDao berthsDao;
    @Override
    public void updateFreeBerthsByPlNo(PushFreeBerthsChangeRequest freeBerthsChangeMsgVO) {
        for (FreeBerthsChangeMsgVO vo : freeBerthsChangeMsgVO.getBerthsChangeMsgVOs()) {
            updateFreeBerthToDB(vo.getPlNo(),vo.getFreeBerths(),vo.getRealFreeBerths());
        }
    }

    @Override
    public Integer updateFreeBerthNumByPlNo(String plNo) {
        logger.info("更新空闲车位数"+plNo);
        //更新该停车场编号的总空闲车位
        Integer totalfreeBerth = Optional.ofNullable(berthsDao.queryFreeBerthNum(plNo,null)).orElse(0L).intValue();
        this.pushFreeBerthsChange(plNo, null, totalfreeBerth);
        //查询空闲车位规则表
        List<TpPFreeberthStatisticRule> freeberthStatisticRules = statisticRuleDao.getFreeStatisticRule(plNo);
        if(CollectionUtils.isEmpty(freeberthStatisticRules)){
            return totalfreeBerth;
        }
        //根据规则统计空闲车位
        for (TpPFreeberthStatisticRule tpPFreeberthStatisticRule : freeberthStatisticRules) {
            String ruleNo = tpPFreeberthStatisticRule.getStatisticRuleNo();
            String sql = tpPFreeberthStatisticRule.getStatisticRuleSql();
            logger.info("根据规则统计空闲车位更新空闲车位数:"+ruleNo);
            Long ruleFreeBerths = berthsDao.queryFreeBerthNum(plNo,sql);
            this.pushFreeBerthsChange(ruleNo, null, ruleFreeBerths.intValue());
        }
        return totalfreeBerth;
    }
    private void pushFreeBerthsChange(String plNo,String plName,Integer freeBerths){
        try{
            //更新缓存
            ParkFreeBerthsCacheUtil.setFreeBerthsByPlNo(plNo, freeBerths);
            //更新数据库
            this.updateFreeBerthToDB(plNo,freeBerths,freeBerths);
        }catch (Exception e) {
            logger.error("空闲车位变化更新失败",e);
        }
    }

    /**
     * 推送空闲车位到诱导屏
     * @param plNo
     * @param freeBerths
     * @param realFreeBerths
     */
    private void pushFreeBerthsToScreen(String plNo, Integer freeBerths,Integer realFreeBerths) {
        logger.debug("开始将空闲车位推送到诱导屏=="+plNo+":"+freeBerths+":"+realFreeBerths);
        String freeberthsRuls = SysCodeValueCacheUtil.queryCodeValue(CodeValKindEnum.FREE_BERTHS_PUSH_RULE);
        List<PushFreeBerthsRule> pushRuls = JSON.parseArray(freeberthsRuls, PushFreeBerthsRule.class);
        for (PushFreeBerthsRule pushFreeBerthsRule : pushRuls) {
            //判断停车场是否满足该推送规则正则表达式
            String regEx = pushFreeBerthsRule.getPlNoRegular();
            Pattern pat = Pattern.compile(regEx);
            Matcher matcher = pat.matcher(plNo);
            boolean rs = matcher.matches();
            if(!rs){
                continue;
            }
            //匹配成功，推送
            switch (pushFreeBerthsRule.getPushFactoryNo()) {
                case QING_DAD:
                    //推送青岛诱导
                    this.qingDaoPuthFreeBerths(plNo,freeBerths,pushFreeBerthsRule);
                    break;
                case EPS:
                    //推送EPS诱导-推送真实的空闲车位数
                    this.epsPushFreeBerths(plNo,realFreeBerths,pushFreeBerthsRule);
                    break;
                default:
                    logger.error("该推送厂家没有找到对应的实现");
                    break;
            }
        }
    }

    /**
     * 更新空闲车位数到数据库
     * @param plNo
     * @param freeBerths
     * @param realFreeBerths
     */
    private void updateFreeBerthToDB(String plNo,Integer freeBerths,Integer realFreeBerths) {
        ParkFreeBerthExample example = new ParkFreeBerthExample();
        example.createCriteria()
                .andPlNoEqualTo(plNo)
                .andDataStatusEqualTo(DataStatusEnum.DATA_STATUS_VALID.value());
        ParkFreeBerth record = new ParkFreeBerth();
        record.setFreeBerth(freeBerths);
        record.setRealFreeBerth(realFreeBerths);
        record.setModifyTime(new Date());
        int count = parkFreeBerthMapper.updateByExampleSelective(record, example);
        if (count == 0) {
            /**没有则插入*/
            this.insertFreeBerthsByPlNo(plNo, freeBerths,realFreeBerths);
        }
        //推送到诱导
        this.pushFreeBerthsToScreen(plNo,freeBerths,realFreeBerths);
    }
    /**
     * 根据停车场编号插入空闲车位数表
     *
     * @param plNo
     * @param freeBerths 2017年7月24日 zhaowg
     */
    private void insertFreeBerthsByPlNo(String plNo, Integer freeBerths,Integer realFreeBerths) {
        //先查询是否已经存在
        ParkFreeBerthExample example = new ParkFreeBerthExample();
        example.createCriteria().andPlNoEqualTo(plNo)
                .andDataStatusEqualTo(DataStatusEnum.DATA_STATUS_VALID.value());
        List<ParkFreeBerth> parkFreeBerths = parkFreeBerthMapper.selectByExample(example );
        if (!CollectionUtils.isEmpty(parkFreeBerths)) {
            return;
        }
        //不存在，则插入停车场空闲车位信息
        ParkFreeBerth record = new ParkFreeBerth();
        record.setDataStatus(DataStatusEnum.DATA_STATUS_VALID.value());
        record.setFreeBerth(freeBerths);
        record.setRealFreeBerth(realFreeBerths);
        record.setModifyTime(new Date());
        record.setPlNo(plNo);
        ParkingLot parkingLot = parkingLotDao.selectByPlNo(plNo);
        if(parkingLot!=null){
            record.setPlId(parkingLot.getPlId());
            record.setPlName(parkingLot.getPlName());
        }
        parkFreeBerthMapper.insertSelective(record);
    }
    /**
     * EPS推送诱导
     * @param plNo
     * @param realFreeBerths
     * 2018年1月22日 zhaowg
     */
    private void epsPushFreeBerths(String plNo, Integer realFreeBerths,PushFreeBerthsRule pushFreeBerthsRule ) {
        if(StringUtils.isEmpty(plNo)){
            return;
        }
        ParkingLot parkingLotDTO = parkingLotDao.selectByPlNo(plNo);
        List<Object> list  = new ArrayList<>();
        Map<String, Object> params = new HashMap<>();
        params.put("freePlaces", realFreeBerths);                          //空余车位-推送真实的空闲车位数
        if(parkingLotDTO.getDataStatus() == 1){
            params.put("garageStatus", "AVAILABLE");                   //停车场状态-生效
        }else{
            params.put("garageStatus", "NOT AVAILABLE");               //停车场状态-失效
        }
        params.put("id", parkingLotDTO.getPlId()==null?1:parkingLotDTO.getPlId().intValue()); //停车场id
        params.put("maxPlaces", parkingLotDTO.getPlBerthNum());        //es总车位数
        params.put("name", parkingLotDTO.getPlName());                 //停车场名称
        params.put("occupancyUpdate", this.getNowTimeStamp());         //时间戳
        list.add(params);
        Map<String, Object> newparams = new HashMap<>();
        newparams.put("garages", list);
        logger.info("--------------为EPS推送的数据为："+ JSONObject.toJSONString(newparams)+"----------EPS推送地址为："+pushFreeBerthsRule.getPushUrl());
        String freeBerth = JSONObject.toJSONString(newparams);
        JSONObject jsonObject = JSONObject.parseObject(freeBerth);
        try {
            String rs = HttpUtil.ignorePostForHttps(pushFreeBerthsRule.getPushUrl(), jsonObject, true);
            logger.info("EPS推送的停车场空闲车位响应信息："+rs);
        } catch (IOException e) {
            throw new BizException(ErrorType.CALL_THIRD_SERVICE_EXCEPTION, e, "EPS推送失败");
        }
    }


    /**
     * 为青岛推送诱导
     * @param plNo
     * @param freeBerths
     * 2018年1月22日 zhaowg
     * @throws UnsupportedEncodingException
     */
    private void qingDaoPuthFreeBerths(String plNo, Integer freeBerths,PushFreeBerthsRule pushFreeBerthsRule ){
        List<InductionRelease> list = new ArrayList<>();
        //传固定值
        InductionRelease inductionRelease = new InductionRelease();
        inductionRelease.setOwner_code("1001");
        inductionRelease.setRange_code(plNo);
        inductionRelease.setBerth_left(String.valueOf(freeBerths));
        inductionRelease.setParking_code(plNo);
        list.add(inductionRelease);
        //开始调用二级诱导批量发布接口

        Map<String, Object> params = new HashMap<>();
        params.put("owner_code", "");
        params.put("pwd", "");
        params.put("jsonStr", JSONObject.toJSONString(list));
        logger.info("青岛推送空闲车位请求参数:" + JSONObject.toJSONString(params)+"----------青岛推送地址为："+pushFreeBerthsRule.getPushUrl());
        try{
            String rs = HttpClientTutorial.httpPostRequest(pushFreeBerthsRule.getPushUrl(), params);
            logger.info("青岛推送空闲车位响应信息："+rs);
            InductionResponse<?> inductionResponse = JSONObject.parseObject(rs, InductionResponse.class);
            String code = inductionResponse.getSys_code();
            if("100".equals(code)){
                logger.info("青岛推送空闲车位成功");
            }else{
                throw new BizException(ErrorType.CALL_THIRD_SERVICE_EXCEPTION, "青岛推送失败："+inductionResponse.getSys_msg(),false);
            }
        }catch (Exception e) {
            throw new BizException(ErrorType.CALL_THIRD_SERVICE_EXCEPTION, e, "青岛推送失败");
        }
    }
    /**
     * 取得当前时间戳（精确到秒）
     *
     * @return nowTimeStamp
     */
    private String getNowTimeStamp() {
        long time = System.currentTimeMillis();
        String nowTimeStamp = String.valueOf(time / 1000);
        return nowTimeStamp;
    }
}
