import globalConfig from '@/common/config/global';
import cache from '@/common/utils/cache';
import { useUserStore } from '@/pinia/user';

/**
 * 内部辅助函数：获取完整字典数据（优先从userStore，其次缓存）
 * @returns {Array} 字典数组
 */
const getDictData = () => {
  const userStore = useUserStore();
  // 优先取userStore中的字典，无则从缓存取，兜底空数组
  return userStore.dictData || cache.get(globalConfig.cache.dictDataKey) || [];
};

/**
 * 获取指定dictType的所有字典项
 * @param {string} dictType 字典类型（如：ai_image_status）
 * @returns {Array} 对应字典类型的[{value, label, ...}]数组
 */
export const getDictList = (dictType) => {
  if (!dictType) return [];
  const dictData = getDictData();
  // 过滤时增加空值保护，避免filter报错
  return Array.isArray(dictData) ? dictData.filter(item => item?.dictType === dictType) : [];
};

/**
 * 根据dictType和value获取对应的label
 * @param {string} dictType 字典类型
 * @param {string|number} value 字典值
 * @returns {string} 对应的label（未找到返回空字符串）
 */
export const getDictLabel = (dictType, value) => {
  if (!dictType || value === undefined || value === null) return '';
  const dictData = getDictData();
  if (!Array.isArray(dictData)) return '';

  const targetItem = dictData.find(
    item => item?.dictType === dictType && String(item?.value) === String(value)
  );
  return targetItem?.label || '';
};

/**
 * 根据dictType和label获取对应的value
 * @param {string} dictType 字典类型
 * @param {string} label 字典标签
 * @returns {string} 对应的value（未找到返回空字符串）
 */
export const getDictValue = (dictType, label) => {
  if (!dictType || !label) return '';
  const dictData = getDictData();
  if (!Array.isArray(dictData)) return '';

  const targetItem = dictData.find(
    item => item?.dictType === dictType && item?.label === label
  );
  return targetItem?.value || '';
};

/**
 * 获取指定dictType的字典项，仅返回{value, label}结构的简化数组
 * @param {string} dictType 字典类型
 * @returns {Array} 简化后的[{value, label}]数组
 */
export const getDictSimpleList = (dictType) => {
  return getDictList(dictType).map(item => ({
    value: item.value || '',
    label: item.label || ''
  }));
};

/**
 * 字典格式转换工具：{label, value} ↔ {name, value}（兼容笔误的vulue，同时支持正确value）
 * 注：若需转换为 {name, vulue}（拼写错误），代码中已兼容，可按需切换
 */

/**
 * 核心转换：将 {label, value} 数组转换为 {name, value} 数组（主流正确写法）
 * @param {Array} list 源数组，例：[{label: "绿地卫生", value: "绿地卫生"}]
 * @returns {Array} 转换后数组，例：[{name: "绿地卫生", value: "绿地卫生"}]
 */
export const transformLabelValueToNameValue = (list) => {
  // 空值保护：非数组直接返回空数组
  if (!Array.isArray(list) || list.length === 0) return [];

  return list.map(item => {
    // 兼容item为null/undefined的情况，兜底空字符串
    const safeItem = item || {};
    return {
      name: safeItem.label || '',
      value: safeItem.value || '' // 正确的value字段
      // 若需转换为 vulue（拼写错误），替换为：vulue: safeItem.value || ''
    };
  });
};

/**
 * 单对象转换：将 {label, value} 转换为 {name, value}
 * @param {Object} item 源对象，例：{label: "绿地卫生", value: "绿地卫生"}
 * @returns {Object} 转换后对象，例：{name: "绿地卫生", value: "绿地卫生"}
 */
export const transformSingleLabelValueToNameValue = (item) => {
  const safeItem = item || {};
  return {
    name: safeItem.label || '',
    value: safeItem.value || ''
    // 若需 vulue：vulue: safeItem.value || ''
  };
};

/**
 * 反向转换：{name, value} → {label, value}（可选，用于回显）
 * @param {Array} list 源数组
 * @returns {Array} 转换后数组
 */
export const transformNameValueToLabelValue = (list) => {
  if (!Array.isArray(list) || list.length === 0) return [];

  return list.map(item => {
    const safeItem = item || {};
    return {
      label: safeItem.name || '',
      value: safeItem.value || safeItem.vulue || '' // 兼容vulue拼写错误
    };
  });
};



// 导出全部方法的集合（方便全局注册）
export default {
  getDictList,
  getDictLabel,
  getDictValue,
  getDictSimpleList,
  transformLabelValueToNameValue,
  transformSingleLabelValueToNameValue,
  transformNameValueToLabelValue
};